<?php
/**
 * Add-ons in Admin
 *
 * Admin functions relating to add-ons.
 *
 * @package    Church_Theme_Content
 * @subpackage Admin
 * @copyright  Copyright (c) 2014 - 2016 churchthemes.com
 * @link       https://github.com/churchthemes/church-theme-content
 * @license    GPLv2 or later
 * @since      1.2
 */

// No direct access
if ( ! defined( 'ABSPATH' ) ) exit;

/*************************************************
 * ADD-ON DATA
 *************************************************/

/**
 * Get all add-ons
 *
 * @since 1.2
 * @return array Addons and their data passed in when registered
 * @global array $ctc_add_ons
 */
function ctc_get_add_ons() {

	global $ctc_add_ons;

	$add_ons = array();

	if ( isset( $ctc_add_ons ) ) {
		$add_ons = $ctc_add_ons;
	}

	return apply_filters( 'ctc_get_add_ons', $add_ons );

}

/**
 * Get single add-on
 *
 * Get single add-on. Optionally get an add-on's argument when specified.
 *
 * @since 1.2
 * @param string Add-on's directory name
 * @param string Optional argument name to get value for
 * @return mixed Arguments, single argument or false
 */
function ctc_get_add_on( $add_on_dir, $arg = false ) {

	$data = false;

	// Get add-ons
	$add_ons = ctc_get_add_ons();

	// Valid add-on?
	if ( isset( $add_ons[$add_on_dir] ) ) {

		// Get add-on data
		$data = $add_ons[$add_on_dir];

		// Get specific argument?
		if ( ! empty( $arg ) ) {

			// Is argument valid? Use value
			if ( isset( $data[$arg] ) ) {
				$data = $data[$arg];
			}

			// If invalid, return false (not array)
			else {
				$data = false;
			}

		}

	}

	return apply_filters( 'ctc_get_add_on', $data, $add_on_dir, $arg );

}

/*************************************************
 * PLUGINS SCREEN
 *************************************************/

/**
 * Add "Add-ons" plugin meta link
 *
 * This will insert an "Add-ons" link into the plugin's meta links (after "View details") on the Plugins screen
 *
 * @since 1.2
 * @param array $plugin_meta An array of the plugin's metadata
 * @param string $plugin_file Path to the plugin file, relative to the plugins directory
 * @param array $plugin_data An array of plugin data
 * @param string $status Status of the plugin
 * @return array Modified meta links
 */
function ctc_add_ons_plugin_meta_link( $plugin_meta, $plugin_file, $plugin_data, $status ) {

	// Modify only this plugin's meta links
	if ( $plugin_file == CTC_FILE_BASE ) {

		// Have links array?
		if ( is_array( $plugin_meta ) ) {

			// Append "Add-ons" link
			$plugin_meta[] = '<a href="' . esc_url( 'https://churchthemes.com/plugins/?utm_source=ctc&utm_medium=plugin&utm_campaign=add-ons&utm_content=plugins' ) . '" target="_blank">' . esc_html__( 'Add-ons', 'church-theme-content' ) . '</a>';

		}

	}

	// Return links array, possibly modified
	return $plugin_meta;

}

add_filter( 'plugin_row_meta', 'ctc_add_ons_plugin_meta_link', 10, 4 );

/**
 * Add "License Key" link to add-on plugin's action links
 *
 * This adds "License Key" link after other action links for each add-on supporting licensing on Plugins screen.
 *
 * @since 1.2
 * @param array $actions An array of plugin action links.
 * @param string $plugin_file Path to the plugin file.
 * @param array $plugin_data An array of plugin data.
 * @param string $context The plugin context.
 * @return array Modified action links
 */
function ctc_add_ons_plugin_meta_links( $actions, $plugin_file, $plugin_data, $context ) {

	$plugin_dirname = dirname( $plugin_file );

	// Is this plugin a CTC add-on that supports licensing
	if ( ctc_edd_license_supported( $plugin_dirname ) ) {

		// Have action links array?
		if ( is_array( $actions ) ) {

			// Append "License Key" link
			$actions[] = '<a href="' . admin_url( 'options-general.php?page=' . CTC_DIR ) . '#licenses">' . esc_html__( 'License Key', 'church-theme-content' ) . '</a>';

		}

	}

	return $actions;

}

add_filter( 'plugin_action_links', 'ctc_add_ons_plugin_meta_links', 10, 4 );

/*************************************************
 * POST FIELDS
 *************************************************/

/**
 * Append Custom Recurring Events note
 *
 * Inform user of add-on by appending note to Recurrence field's description
 *
 * @since 1.2
 * @param array $field Field settings
 * @return array Modified field settings
 */
function ctc_append_custom_recurrence_note( $field ) {

	// Show note only when add-on is not in use
	if ( ! defined( 'CTC_CRE_VERSION' ) ) {

		// Description key may not be set
		$field['desc'] = isset( $field['desc'] ) ? $field['desc'] : '';

		// If has desc, break to new line
		if ( ! empty( $field['desc'] ) ) {
			$field['desc'] .= '<br>';
		}

		// Append note
		// Descriptions are sanitized with wp_kses automatically by CT Meta Box
		$field['desc'] .= sprintf(
			__( 'Install <a href="%1$s" target="_blank">Custom Recurring Events</a> for more options.', 'church-theme-content' ),
			'https://churchthemes.com/plugins/custom-recurring-events/?utm_source=ctc&utm_medium=plugin&utm_campaign=custom_recur_events&utm_content=recurrence_field'
		);

	}

	return $field;

}

add_action( 'ctmb_field-_ctc_event_recurrence', 'ctc_append_custom_recurrence_note' );
